
#include "sockets.h"
#include "cmfx.h"
#include "logging.h"
#include "utils.h"
#include "timing.h"
#include "commands.h"

SOCKTYPE listen_sock = 0;
descrdata *sockets = 0;
int max_socket = 0;
unsigned int listen_port = 8888;
time_t sock_wait = 0;

int sock_init() {

#ifdef _WIN32
	WORD sockVer;
	WSADATA wsaData;

	sockVer = MAKEWORD(2,0);

	if (0 != WSAStartup(sockVer, &wsaData)) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Unable to start windows socket layer.\r\n");
		return -1;
	}

	if ( LOBYTE( wsaData.wVersion ) != 2 ||	HIBYTE( wsaData.wVersion ) != 0 ) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Windows sockets is not version 2.0.\r\n");
		WSACleanup();
		return -1;
	}
#endif

	const char *port_str = cmfx_get_var("PORT");
	listen_port = (port_str ? (atoi(port_str) ? atoi(port_str) : listen_port) : listen_port);

	if ( sock_make_socket(listen_port) != 0) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Could not open listening socket.\r\n");
#ifdef _WIN32
		WSACleanup();
#endif
		return -1;
	}

	// How long should the sockets wait for data?
	const char *wait_str = cmfx_get_var("SOCKETWAIT");
	sock_wait = (wait_str ? atoi(cmfx_get_var("SOCKETWAIT")) : SOCK_DEFAULTWAIT);

	log_log(LOG_NOTICE,"[SOCK] NOTICE: Listening for connections on port %d.\r\n",listen_port);

	return 0;
}

int sock_cleanup() {
	// Close all sockets, and end. Hopefully by now, all output has been handled.
	// Make sure to update the cache first with the current users.
	while (sockets) {
		descrdata *n = sockets->next;
		SOCKCLOSE(sockets->descr);
		sock_forget_descr(sockets->descr);
		sockets = n;
	}
	SOCKCLOSE(listen_sock);

#ifdef _WIN32
	WSACleanup();
#endif

	log_log(LOG_DEBUG,"[SOCK] DEBUG: Socket layer shut down.\r\n");

	return 0;
}

int sock_handle() {
	time_t net = timing_next_event_time();
	sock_wait = ((net - time(NULL)) < sock_wait ? net - time(NULL) : SOCK_DEFAULTWAIT);
	if (sock_wait < 0) sock_wait = 0;
	fd_set in, out;
	
	// Zero out the sets for queuing
	FD_ZERO(&in);
	FD_ZERO(&out);

	// Listen for incoming connections on the input socket
	FD_SET(listen_sock,&in);

	descrdata *p = 0;
	descrdata *s = 0;
	while (1) {
		p = s; // Assign the previous pointer

		if (!s) s = sockets; // Setup the list at the head, or the next
		else s = s->next;

		if (!s) break; // If there is no current socket, exit

		if (s->boot) {
			log_log(LOG_NOTICE,"[SOCK] NOTICE: Socket from %s disconnected.\r\n",s->hostname);
			sock_goodbye(s);
			SOCKCLOSE(s->descr);
			s->user = 0; // Loose our reference, the cache will keep it and expire if needed
			if (!p) {
				sockets = s->next;
				sock_forget_descr(s->descr);
				s = 0;
				continue;
			} else {
				p->next = s->next;
				sock_forget_descr(s->descr);
				s = p;
				continue;
			}
		}
		if (s->outbuf) FD_SET(s->descr, &out);
		FD_SET(s->descr,&in);
		if (s->commands) sock_wait = 0; // If there are commands to process, don't wait
	}

	struct timeval timeout;

	if (sock_wait <= 0) {
		timeout.tv_sec = 0;
		timeout.tv_usec = 1;
	} else {
		timeout.tv_sec = (long) sock_wait;
		timeout.tv_usec = 0;
	}

	log_log(LOG_DEBUG,"[SOCK] DEBUG: Waiting for connections (%d seconds).\r\n",sock_wait);

	if (select(max_socket + 1, &in, &out, (fd_set *) 0, &timeout) == SELECT_ERROR) {
#ifdef _WIN32
        if (WSAGetLastError() != WSAEINTR) {
			log_log(LOG_ERROR,"[SOCK] ERROR: Death during select.\r\n");
			return -1;
        }
#else
        if (errno != EINTR) {
            log_log(LOG_ERROR,"[SOCK] ERROR: Death during select.\r\n");
            return -1;
        }
#endif
	} else {
		if (FD_ISSET(listen_sock, &in)) {
			sock_new_conn(listen_port, listen_sock);
		}
		for (descrdata *s = sockets; s; s = s->next ) {
			if (FD_ISSET(s->descr, &in)) {
				if (sock_procin(s)) {
					s->boot = 1;
				}
			}
			if (FD_ISSET(s->descr, &out)) {
				if (sock_procout(s)) {
					s->boot = 1;
				}
			}
			cmd_proc_commands(s);
		}
	}

	return 0;
}

void sock_write(descrdata *s, const char *str) {
	if (!s) return;
	if (!str) return;

	if (!s->outbuf) {
		s->outbuf = (char *) zmalloc(MAX_SEND + 1);
		s->outbufend = s->outbuf;
	}

	const char *p = str;
	while (s->outbufend < s->outbuf + MAX_SEND && *p) {
		*s->outbufend++ = *p++;
	}
	*s->outbufend = '\0';
}

void sock_addline(descrdata *s, char *str) {

	inLines *i = s->commands;
	if (!i) {
		s->commands = (inLines *) zmalloc(sizeof(inLines));
		s->commands->line = make_string(str);
		s->commands->next = 0;
		return;
	}
	while (i->next) i = i->next;

	i->next = (inLines *) zmalloc(sizeof(inLines));
	i->next->line = make_string(str);
	i->next->next = 0;

	return;
}

int sock_procin(descrdata *s) {
	char buf[MAX_RECV+1];
	int br = 0;

	memset(buf,'\0',sizeof(buf));

	br = SOCKRECV(s->descr, buf, MAX_RECV, 0);

	if (br < 0) {
#ifdef _WIN32
		if (WSAGetLastError() != WSAEWOULDBLOCK) {
			log_log(LOG_ERROR,"[SOCK] ERROR: Socket receive operation failed.\r\n");
			return -1;
		} else {
			return 0;
		}
#else
      if (errno != EWOULDBLOCK) {
         log_log(LOG_ERROR,"[SOCK] Error: Socket recieve operation failed.\r\n");
         return -1;
      } else {
         return 0;
      }
#endif
	} else if (br == 0) {
		return -1;
	} else {
		buf[br] = '\0';
		if (!s->inbuf) {
			s->inbuf = (char *) zmalloc(MAX_RECV+1);
			s->inbufend = s->inbuf;
		}
		char *cmd, *cmdend, *qin, *qend;
		cmd = s->inbufend;
		cmdend = s->inbuf + MAX_RECV - 1;
		for (qin = buf, qend = buf + br; qin < qend; qin++) {
			if (*qin == '\n') {
				*cmd = '\0';
				if (cmd > s->inbuf) {
					sock_addline(s,s->inbuf);
				}
				free(s->inbuf);
				s->inbuf = (char *) zmalloc(MAX_RECV + 1);
				s->inbufend = s->inbuf;
				cmd = s->inbuf;
			} else if (cmd < cmdend && isascii(*qin)) {
				if (isprint(*qin)) {
					*cmd++ = *qin;
				} else if (*qin == '\t') {
					*cmd++ = ' ';
				} else if (*qin == 8 || *qin == 127) {
					if (cmd > s->inbuf) cmd--;
				} else if (*qin == 27) {
					*cmd++ = 27;
				}
			}
		}
		if (cmd > s->inbuf) {
			s->inbufend = cmd;
		} else {
			free(s->inbuf);
			s->inbuf = 0;
			s->inbufend = 0;
		}
	}

	return 0;
}

int sock_procout(descrdata *s) {
	if (!s) return -1;

	if (!s->outbuf) return -1;

	int bs = SOCKSEND(s->descr, s->outbuf, (int)strlen(s->outbuf), 0);

	if (bs == 0) {
		return -1;
	} else if (bs != strlen(s->outbuf)) {
		char *p = s->outbuf + strlen(s->outbuf) - 1;
		char *n = (char *) zmalloc(strlen(p) + 1);
		strcpy(n,p);
		free(s->outbuf);
		s->outbuf = n;
		return 0;
	} else {
		free(s->outbuf);
		s->outbuf = 0;
		return 0;
	}
}

// Look through the socket sets to find the correct descriptor
descrdata * sock_find_by_descr(SOCKTYPE descr) {
	for (descrdata *s = sockets; s; s = s->next) {
		if (s->descr == descr) return s;
	}

	return 0;
}

descrdata * sock_find_by_vnum(int vnum) {
	for (descrdata *s = sockets; s; s = s->next) {
		if (s->user) {
			if (s->user->vnum == vnum) {
				return s;
			}
		}
	}
	return 0;
}

descrdata * sock_find_by_name(const char *name) {
	for (descrdata *s = sockets; s; s = s->next) {
		if (s->user) {
			if (!STRICMP(s->user->name,name)) {
				return s;
			}
		}
	}

	return 0;
}


// Look through the socket sets to find the correct descriptor
descrdata * sock_find_by_conn(unsigned int conn) {
	for (descrdata *s = sockets; s; s = s->next) {
		if (s->conn == conn) return s;
	}

	return 0;
}

int sock_make_socket(int port) {
	SOCKTYPE s;
	int opt;
	struct sockaddr_in server;

	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0) return -1;

	opt = 1;
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *) &opt, sizeof(opt)) < 0) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Could not set socket options.\r\n");
		return -1;
	}

	server.sin_family = AF_INET;
	server.sin_addr.s_addr = INADDR_ANY;
	server.sin_port = htons(port);

	if (bind(s, (struct sockaddr *) &server, sizeof(server))) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Could not bind socket.\r\n");;
		SOCKCLOSE(s);
		return -1;
	}

	listen(s, SOMAXCONN);

	listen_sock = s;

	max_socket = ((int)s > max_socket ? (int)s : max_socket);

	log_log(LOG_DEBUG,"[SOCK] DEBUG: Listen port on %d is open.\r\n",port);

	return 0;
}

void sock_new_conn(int p, SOCKTYPE s) {
	SOCKTYPE new_sock;
	struct sockaddr_in addr;

#ifdef _WIN32
	int addr_len;
#else
   unsigned int addr_len;
#endif	
	char hostname[128];

	addr_len = sizeof(addr);

	new_sock = accept(s, (struct sockaddr *) &addr, &addr_len);
	if (new_sock == ACCEPT_ERROR) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Could not accept new connection.\r\n");
		return;
	}
	strcpy(hostname, sock_addrout(p, addr.sin_addr.s_addr, addr.sin_port));

	// Make the socket non-blocking
#ifdef _WIN32
	unsigned long O_NONBLOCK = 1;
	if (ioctlsocket(new_sock, FIONBIO, &O_NONBLOCK) == SOCKET_ERROR) {
		log_log(LOG_ERROR,"[SOCK] ERROR: Failed to make socket nonblocking.\r\n");
		SOCKCLOSE(new_sock);
		return;
	}
#else
   if (fcntl(new_sock, F_SETFL, O_NONBLOCK) == -1) {
      log_log(LOG_ERROR,"[SOCK] ERROR: Failed to make socket nonblocking.\r\n");
      SOCKCLOSE(new_sock);
      return;
   }
#endif

	// Add the socket into the chain
	descrdata *d = sock_remember_descr(new_sock, hostname);
	
   max_socket = ((int)new_sock > max_socket ? (int)new_sock : max_socket);	

	log_log(LOG_NOTICE,"[SOCK] NOTICE: Accepted connection from [%d|%d] %s.\r\n",d->conn,new_sock,hostname);

	sock_welcome(d);

	return;
}

void sock_goodbye(descrdata *d) {
	sock_write(d,"Goodbye.\r\n");
	sock_procout(d);
	descrdata *o = sockets;
	while (o) {
		if (o->status == STATUS_LOGGEDIN) {
			if (o != d) {
				sock_write(o,d->user->name);
				sock_write(o," has disconnected.\r\n");
			}
		}
		o = o->next;
	}
}

void sock_welcome(descrdata *s) {
	struct STATSTRUCT sta;
	if (!STAT("welcome.txt",&sta)) {
		char *buf = (char *) zmalloc(sta.st_size + 1);
		FILE *welcome = fopen("welcome.txt","rb");
		if (welcome) {
			int bytes = (int) fread(buf,sizeof(char),sta.st_size,welcome);
			buf[bytes] = '\0';
			fclose(welcome);
			sock_write(s,buf);
			free(buf);
		} else {
			sock_write(s,"welcome.txt is missing\r\n");
		}
	} else {
		sock_write(s,"welcome.txt is missing\r\n");
	}

	sock_write(s, "\r\nEnter a username: ");
	s->status = STATUS_WELCOMED;
}

// Add the socket to the connected chain
descrdata * sock_remember_descr(SOCKTYPE s, char hostname[128]) {
	unsigned int conn = 1;
	descrdata *sk = (descrdata *) zmalloc(sizeof(descrdata));
	sk->descr     = s;
	sk->next      = 0;
	sk->conn      = conn;
	sk->outbuf    = 0;
	sk->outbufend = 0;
	sk->commands  = 0;
	sk->inbuf     = 0;
	sk->inbufend  = 0;
	sk->boot      = 0;
	sk->status    = 0;
	strcpy(sk->hostname,hostname);

	if (!sockets) {
		sockets = sk;
		return sk;
	} else {
		descrdata *prev = 0;
		descrdata *curr = sockets;
		while (curr) {
			if (conn < curr->conn) {
				// If no previous, we're at the head
				if (!prev) {
					sk->next = sockets;
					sockets = sk;
					sk->conn = conn;
					return sk;
				}
				sk->next = curr;
				prev->next = sk;
				sk->conn = conn;
				return sk;
			}
			conn++;
			prev = curr;
			curr = curr->next;
		}
		prev->next = sk;
		sk->conn = conn;
		return sk;
	}
}

// Remove the socket from the connected chain
void sock_forget_descr(SOCKTYPE sock) {
	for (descrdata *s = sockets, *p = 0; s ; p = s, s = s->next) {
		if (s->descr == sock) {
			if (s == sockets) {
				sockets = s->next;
				if (s->outbuf) free(s->outbuf);
				if (s->inbuf) free(s->inbuf);
				while (s->commands) {
					if(s->commands->line) free(s->commands->line);
					s->commands = s->commands->next;
				}
				free(s);
			} else {
				p->next = s->next;
				if (s->outbuf) free(s->outbuf);
				if (s->inbuf) free(s->inbuf);
				while (s->commands) {
					if(s->commands->line) free(s->commands->line);
					s->commands = s->commands->next;
				}
				free(s);
			}
			return;
		}
	}

}

// Convert the IP address to a DNS name
char * sock_addrout(int lport, long a, unsigned short prt) {
	static char buf[128];
	struct in_addr addr;

	addr.s_addr = a;
	prt = ntohs(prt);

	struct hostent *he = gethostbyaddr(((char *) &addr), sizeof(addr), AF_INET);
	if (he) {
		sprintf(buf, "%s(%u)", he->h_name, prt);
		return buf;
	}

	a = ntohl(a);

	sprintf(buf, "%ld.%ld.%ld.%ld(%u)", (a >> 24) & 0xff, (a >> 16) & 0xff, (a >> 8) & 0xff, a & 0xff, prt);

	return buf;
}

