
#include "utils.h"
#include "logging.h"
#include "cmfx.h"
#include "db_gen.h"
#include "db_mysql.h"
#include "objects.h"

MYSQL * mysql_handle = 0;


int db_mysql_init() {

	mysql_handle = mysql_init((MYSQL *) 0);

	if (!mysql_handle) {
		log_log(LOG_ERROR,"[SQL ] ERROR: Couldn't startup the MySQL layer.\r\n");
		return -1;
	}

	const char *dbhost = cmfx_get_var("DBHOST");
	if (!dbhost) dbhost = "localhost";
	const char *dbuser = cmfx_get_var("DBUSER");
	if (!dbuser) dbuser = "cmfx";
	const char *dbpass = cmfx_get_var("DBPASS");
	if (!dbpass) dbpass = "cmfx";
	const char *dbdb = cmfx_get_var("DBDB");
	if (!dbdb) dbdb = "cmfx";
	const char *sdbport = cmfx_get_var("DBPORT");
	if (!sdbport) sdbport = "";
	int dbport = atoi(sdbport);


	// MYSQL *, hostname, username, password, database, ...
	if (!mysql_real_connect(mysql_handle, dbhost, dbuser, dbpass, dbdb, dbport, 0 ,0)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: Couldn't connect to the database.\r\n");
		return -1;
	}

	// Get the 'max' object vnum from the database for the 'top' value
	if (mysql_query(mysql_handle,"SELECT MAX(vnum) FROM objects")) {
		log_log(LOG_ERROR,"[SQL ] ERROR: Couldn't retreive top database number.\r\n");
		return -1;
	}

	MYSQL_RES *result = mysql_store_result(mysql_handle);
	MYSQL_ROW row = 0;
	if (!(row = mysql_fetch_row(result))) {
		log_log(LOG_ERROR,"[SQL ] ERROR: Database appears to empty!\r\n");
		return -1;
	}

	db_top = atoi(row[0]);
	mysql_free_result(result);

	log_log(LOG_DEBUG,"[SQL ] DEBUG: MySQL layer is up and running.\r\n");

	return 0;
}

PROP_LIST * db_mysql_get_props(int vnum) {
	char *query = (char *)zmalloc(54 + intlen(vnum) + 1);
	sprintf(query,"SELECT * FROM properties WHERE vnum=%d ORDER BY property",vnum);

	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error retrieving object properties by vnum.\r\n");
		return 0;
	}

	MYSQL_RES *result = mysql_store_result(mysql_handle);
	if (!result) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: No properties for vnum %d in database.\r\n",vnum);
		return 0;
	}

	int num_matches = (int) mysql_num_rows(result);
	if (num_matches < 1) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: No properties for vnum %d in database.\r\n",vnum);
		mysql_free_result(result);
		return 0;
	}

	PROP_LIST *head = 0;
	PROP_LIST *pl = head;

	MYSQL_ROW row = 0;
	while ((row = mysql_fetch_row(result))) {
		if (!pl) {
			pl = head = (PROP_LIST *) zmalloc(sizeof(PROP_LIST));
		} else {
			pl->next = (PROP_LIST *) zmalloc(sizeof(PROP_LIST));
			pl = pl->next;
		}
		PROPERTY *p = (PROPERTY *) zmalloc(sizeof(PROPERTY));
		p->property = make_string(row[1]);
		p->value = make_string(row[2]);
		pl->prop = p;
		pl->next = 0;
	}

	mysql_free_result(result);

	return head;
}

void db_mysql_save_obj(DATA_OBJ *obj) {

	if (!obj) return;

	// Delete the old object, rather than doing an update on all of it.
	char *query = (char *)zmalloc(31 + intlen(obj->vnum) + 1);
	sprintf(query,"DELETE FROM objects WHERE vnum=%d",obj->vnum);

	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error saving an object.\r\n");
		free(query);
		return;
	}
	free(query);

	char *new_name = escape_quotes(obj->name);
	// Reinsert the object
	query = (char *)zmalloc(44 + intlen(obj->vnum) + strlen(new_name) + intlen(obj->flags) + intlen(obj->security)
		+ intlen(obj->parent) + intlen(obj->spawntime) + intlen(obj->location) + intlen(obj->home) + intlen(obj->owner)
		+ intlen(obj->value) + intlen(obj->created) + intlen(obj->modified) + intlen(obj->contents) + intlen(obj->exits)
		+ intlen(obj->next) + 1);
	sprintf(query,"INSERT INTO objects VALUES(%d,'%s',%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d)",
		obj->vnum,new_name,obj->flags,obj->security,obj->parent,obj->spawntime,obj->location,
		obj->home,obj->owner,obj->value,obj->created,obj->modified,obj->contents,obj->exits,obj->next);

	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error saving an object.\r\n");
		free(query);
		return;
	}
	free(query);

	// Delete the old properties
	query = (char *)zmalloc(34 + intlen(obj->vnum) + 1);
	sprintf(query,"DELETE FROM properties WHERE vnum=%d",obj->vnum);

	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error saving an object.\r\n");
		free(query);
		return;
	}
	free(query);

	// Now reinsert all the new properties
	PROP_LIST *pl = obj->props;
	while (pl) {
		if (pl->prop) {
			char *new_val = escape_quotes(pl->prop->value);
			char *new_pro = escape_quotes(pl->prop->property);
			query = (char *)zmalloc(38 + 
				intlen(obj->vnum) + strlen(new_pro) +
				strlen(new_val) + 1);
			sprintf(query,"INSERT INTO properties VALUES(%d,'%s','%s')",obj->vnum,new_pro,new_val);

			if (mysql_query(mysql_handle, query)) {
				log_log(LOG_ERROR,"[SQL ] ERROR: There was an error saving an object's property.\r\n");
			}
			free(query);
		}
		pl = pl->next;
	}

}

DATA_OBJ * db_mysql_load_obj(int vnum) {
	if (vnum < 0) return 0;

	char *query = (char *)zmalloc(33 + intlen(vnum) + 1);
	sprintf(query,"SELECT * FROM objects WHERE vnum=%d",vnum);

	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error retrieving object by vnum.\r\n");
		free(query);
		return 0;
	}

	MYSQL_RES *result = mysql_store_result(mysql_handle);
	if (!result) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: No match for vnum %d in databse.\r\n",vnum);
		free(query);
		return 0;
	}

	int num_matches = (int) mysql_num_rows(result);
	if (num_matches > 1) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: Ambiguous match for vnum %d.\r\n",vnum);
		mysql_free_result(result);
		free(query);
		return 0;
	} else if (num_matches < 1) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: No match for vnum %d in database.\r\n",vnum);
		mysql_free_result(result);
		free(query);
		return 0;
	}

	MYSQL_ROW row = mysql_fetch_row(result);

	DATA_OBJ *dao = (DATA_OBJ *) zmalloc(sizeof(DATA_OBJ));
	dao->vnum = atoi(row[0]);
	strncpy(dao->name,row[1],255);
	dao->flags = (unsigned int) atoi(row[2]);
	dao->security = (unsigned short) atoi(row[3]);
	dao->parent = atoi(row[4]);
	dao->spawntime = atoi(row[5]);
	dao->location = atoi(row[6]);
	dao->home = atoi(row[7]);
	dao->owner = atoi(row[8]);
	dao->value = atoi(row[9]);
	dao->created = atol(row[10]);
	dao->modified = atol(row[11]);
	dao->contents = atoi(row[12]);
	dao->exits = atoi(row[13]);
	dao->next = atoi(row[14]);
	dao->props = db_mysql_get_props(vnum);


	mysql_free_result(result);
	free(query);

	return dao;

}

DATA_OBJ * db_mysql_load_pc_name(const char *name) {
	if (!name) return 0;

	// 103 is the len of the SQL query, 1 is for the null
	char *query = (char *)zmalloc(56 + strlen(name) + 1);
	sprintf(query,"SELECT vnum FROM objects WHERE name = '%s' AND (flags & 1)",name);

	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error executing a select PC by name query.\r\n");
		printf("%s\r\n",mysql_error(mysql_handle));
		return 0;
	}

	MYSQL_RES *result = mysql_store_result(mysql_handle);
	if (!result) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: No match for user %s in databse.\r\n",name);
		return 0;
	}

	int num_matches = (int) mysql_num_rows(result);
	if (num_matches > 1) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: Ambiguous match for user %s.\r\n",name);
		mysql_free_result(result);
		return 0;
	} else if (num_matches < 1) {
		log_log(LOG_DEBUG,"[SQL ] DEBUG: No match for user %s in database.\r\n",name);
		mysql_free_result(result);
		return 0;
	}

	MYSQL_ROW row = mysql_fetch_row(result);
	const char *vnum_str = row[0];
	int vnum = (vnum_str == 0 ? 0 : atoi(vnum_str));

	mysql_free_result(result);

	DATA_OBJ *char_obj = db_mysql_load_obj(vnum);

	if (!char_obj) {
		log_log(LOG_ERROR,"[SQL ] ERROR: No vnum match for user that exists!\r\n");
		return 0; // If there's no object associted, something is freaky!
	}

	return char_obj;
}


DATA_OBJ * db_mysql_get_garbage_obj() {
	char *query = (char *) zmalloc(40 + intlen(OBJ_TYPE_GARBAGE) + 4 + intlen(OBJ_TYPE_GARBAGE) + 1);
	sprintf(query, "SELECT vnum FROM objects WHERE (flags & %d) = %d", OBJ_TYPE_GARBAGE, OBJ_TYPE_GARBAGE);
	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error selecting a garbage object from the database.\r\n");
		free(query);
		return 0;
	}
	free(query);

	MYSQL_RES *result = mysql_store_result(mysql_handle);
	if (result) {
		int num_matches = (int) mysql_num_rows(result);
		if (num_matches > 0) {
			MYSQL_ROW row = mysql_fetch_row(result);
			const char *vnum_str = row[0];
			int vnum = (vnum_str == 0 ? 0 : atoi(vnum_str));
			mysql_free_result(result);
			
			// We now have the vnum of an object to create. Update the flags on the
			// the database object, and return this old garbage to the caller
			char *query = (char *) zmalloc(37 + intlen(OBJ_TYPE_GARBAGE) + 15 + intlen(vnum) + 1);
			sprintf(query, "UPDATE objects SET flags = (flags & ~%d) WHERE vnum = %d", OBJ_TYPE_GARBAGE, vnum);
			if (mysql_query(mysql_handle, query)) {
				log_log(LOG_ERROR,"[SQL ] ERROR: There was an error updating the garbage object in the database.\r\n");
				return 0;
			}
			
			return db_mysql_load_obj(vnum);

		}
	}

	log_log(LOG_DEBUG,"[SQL ] DEBUG: No garbage objects in the database.\r\n");

	mysql_free_result(result);
	// There wasn't a free object, so we need to create one in the database to return
	// The db_gen wrapper will initialize all the default values for us

	db_top++; // We do this because now db_top is a new item, guarenteed

	query = (char *) zmalloc( 27 + intlen(db_top) + 41 + 1);
	sprintf(query, "INSERT INTO objects VALUES(%d,'##GARBAGE##',0,0,0,0,0,0,0,0,0,0,0,0,0)", db_top);
	if (mysql_query(mysql_handle, query)) {
		log_log(LOG_ERROR,"[SQL ] ERROR: There was an error creating a new object in the database.\r\n");
		return 0;
	}

	return db_mysql_load_obj(db_top);

}
void db_mysql_cleanup() {

	if (mysql_handle) mysql_close(mysql_handle);

	log_log(LOG_DEBUG,"[SQL ] DEBUG: MySQL layer has been shut down.\r\n");

}