
#include "cmfx.h"
#include "logging.h"
#include "utils.h"
#include "sockets.h"
#include "db_gen.h"
#include "db_mysql.h"
#include "objects.h"

CACHE_OBJ *obj_cache = 0;
int db_top = -1;

int db_gen_init() {
	const char *dbtype = cmfx_get_var("DBTYPE");
	if (!dbtype) dbtype = "MYSQL";

	if (!STRICMP(dbtype,"mysql")) return db_mysql_init();
//	if (!STRICMP(dbtype,"file")) return db_file_init();

	return -1;
}

void db_gen_cleanup() {
	const char *dbtype = cmfx_get_var("DBTYPE");
	if (!dbtype) dbtype = "MYSQL";

	if (!STRICMP(dbtype,"mysql")) return db_mysql_cleanup();
//	if (!STRICMP(dbtype,"file")) return db_file_cleanup();

}

void db_gen_flush_cache() {
	while (obj_cache) {
		db_gen_save_obj(obj_cache->obj);
		db_gen_free_obj(obj_cache->obj);
		CACHE_OBJ *co = obj_cache;
		obj_cache = obj_cache->next;
		free(co);
	}
	obj_cache = 0;
}

void db_gen_clean_cache() {
   CACHE_OBJ *curr = obj_cache, *prev = 0;
   
   time_t now = time(NULL);
   const char *maxagestr = cmfx_get_var("MAXAGE");
   time_t maxage = (maxagestr ? atoi(maxagestr) : DB_DEFMAXAGE);
   
   log_log(LOG_DEBUG,"[DBGN] DEBUG: Starting cache expirery.\r\n");   

   while (curr) {
      // If the object is past it's maximum cache age, and it's not a player,
      // the free the object from the cache chain.
      if (now - curr->accessed_time >= maxage && !sock_find_by_vnum(curr->obj->vnum)) {
         log_log(LOG_DEBUG,"[DBGN] DEBUG: Expiring cached object %s(%d)\r\n", curr->obj->name, curr->obj->vnum);
         if (!prev) {
            db_gen_save_obj(curr->obj);
		      db_gen_free_obj(curr->obj);
            obj_cache = curr->next;
            CACHE_OBJ *temp = curr;
            curr = curr->next;
            free(temp);
            continue;
         } else {
            db_gen_save_obj(curr->obj);
		      db_gen_free_obj(curr->obj);
            prev->next = curr->next;
            CACHE_OBJ *temp = curr;            
            curr = curr->next;
            free(temp);
            continue;
         }
      }
      prev = curr;
      curr = curr->next;
   }
   log_log(LOG_DEBUG,"[DBGN] DEBUG: Done expiring objects.\r\n");      
}

DATA_OBJ * db_gen_load_obj(int vnum) {

	DATA_OBJ *dpo = 0;

	const char *dbtype = cmfx_get_var("DBTYPE");
	if (!dbtype) dbtype = "MYSQL";

	// Check for the cached object first
	dpo = db_gen_get_cache_obj(vnum);
	if (dpo) {
		if (dpo->flags & OBJ_TYPE_GARBAGE) return 0;
		return dpo;
	}

	if (!STRICMP(dbtype,"mysql")) dpo = db_mysql_load_obj(vnum);
	db_gen_cache_obj(dpo);
	if (dpo->flags & OBJ_TYPE_GARBAGE) return 0;

	return dpo;


}

DATA_OBJ * db_gen_load_pc_name(const char *name) {
	const char *dbtype = cmfx_get_var("DBTYPE");
	if (!dbtype) dbtype = "MYSQL";

	DATA_OBJ *dat = 0;

	// Check the cache for an already cached player.
	dat = db_gen_get_cache_name(name);
	if (dat) return dat;

	if (!STRICMP(dbtype,"mysql")) dat = db_mysql_load_pc_name(name);
	db_gen_cache_obj(dat);

	return dat;

}

void db_gen_del_property(DATA_OBJ *obj, const char *prop) {
	if (!obj) {
		log_log(LOG_WARNING, "[DBGN] WARNING: No object to remove properties for.\r\n");
		return;
	}

	if (!obj->props) {
		return;
	}

	PROP_LIST *curr = obj->props;
	PROP_LIST *prev = 0;
	while (curr) {
		if (curr->prop) {
			int lex = STRICMP(curr->prop->property,prop);
			if (!lex) {
				if (prev) {
					prev->next = curr->next;
					db_gen_free_property(curr->prop);
					free(curr);
					return;
				} else {
					obj->props = obj->props->next;
					db_gen_free_property(curr->prop);
					free(curr);
					return;
				}
			} else if (lex > 0) {
				return;
			}
		}
		prev = curr;
		curr = curr->next;
	}

}

void db_gen_add_property(DATA_OBJ *obj, const char *prop, const char *value) {
	PROPERTY *p = (PROPERTY *) zmalloc(sizeof(PROPERTY));
	p->property = make_string(prop);
	p->value = make_string(value);
	db_gen_add_property(obj,p);
}

// This function links the property pointed to by p into the chain.
// The property should not be freed or go out of scope.
void  db_gen_add_property(DATA_OBJ *obj, PROPERTY *p) {

	if (!obj) {
		log_log(LOG_WARNING, "[DBGN] WARNING: No object to set properties for.\r\n");
		return;
	}

	if (!p) {
		log_log(LOG_WARNING, "[DBGN] WARNING: No property to set on object.\r\n");
		return;
	}

	if (!obj->props) {
		obj->props = (PROP_LIST *) zmalloc(sizeof(PROP_LIST));
		obj->props->prop = p;
		obj->props->next = 0;
		return;
	}

	PROP_LIST *curr = obj->props;
	PROP_LIST *prev = 0;
	while (curr) {
		if (!curr->prop) {
			prev = curr;
			curr = curr->next;
			continue;
		}
		int lex = STRICMP(p->property,curr->prop->property);

		if (!lex) {
			db_gen_free_property(curr->prop);
			curr->prop = p;
			return;
		} else if (lex > 0) {
			prev = curr;
			curr = curr->next;
		} else {
			break;
		}
	}

	if (!prev) {
		PROP_LIST *npl = (PROP_LIST *) zmalloc(sizeof(PROP_LIST));
		npl->next = obj->props;
		obj->props = npl;
		npl->prop = p;
	} else {
		PROP_LIST *npl = (PROP_LIST *) zmalloc(sizeof(PROP_LIST));
		npl->next = curr;
		prev->next = npl;
		npl->prop = p;
	}

}

// If the property is not found, return 0
PROPERTY * db_gen_get_property(DATA_OBJ *obj, const char *prop) {

	// Why someone would want to get props on a non-existant object
	// is beyond me. But, just in case, we'll do so and scream a
	// warning so we know its broken.
	if (!obj) {
		log_log(LOG_WARNING, "[DBGN] WARNING: No object to get props for.\r\n");
		return 0;
	}

	PROP_LIST  *pl = obj->props;

	while (pl) {
		if (pl->prop) {
			int lex = STRICMP(prop,pl->prop->property);
			if (!lex) {
				return pl->prop;
			} else if (lex > 0) {
				pl = pl->next;
			} else {
				pl = 0;
			}
		} else {
			pl = pl->next;
		}
	}

	return 0;
}

void db_gen_free_property(PROPERTY *prop) {
	if (prop) {
		if (prop->property) free(prop->property);
		if (prop->value) free(prop->value);
		free(prop);
	}

}

DATA_OBJ * db_gen_get_cache_name(const char *name) {

	// First check the online player cache -- This saves us a little time
	// as most operations are perfomed on connected users
	descrdata *d = sock_find_by_name(name);
	if (d && d->user) return d->user;

	CACHE_OBJ *p = obj_cache;

	while (p) {
		if (p->obj) {
			if (p->obj->flags & 0x00000001) {
				if (!STRICMP(p->obj->name,name)) {
					return p->obj;
				}
			}
		}
		p = p->next;
	}

	return 0;
}

DATA_OBJ * db_gen_get_cache_obj(int vnum) {

	CACHE_OBJ *p = obj_cache;

	while (p) {
		if (p->obj) {
			if (p->obj->vnum == vnum) {
		      p->accessed_time = time(NULL);					
				return p->obj;
			} else if (vnum > p->obj->vnum) {
				p = p->next;
			} else {
				return 0;
			}
		}
	}

	return 0;
}

void db_gen_cache_obj(DATA_OBJ *obj) {
	if (!obj) return;

	if (!obj_cache) {
		obj_cache = (CACHE_OBJ *) zmalloc(sizeof(CACHE_OBJ));
		obj_cache->obj = obj;
		obj_cache->cache_time = time(NULL);
		obj_cache->accessed_time = time(NULL);		
		obj_cache->next = 0;
		return;
	}

	CACHE_OBJ *prev = 0;
	CACHE_OBJ *curr = obj_cache;
	while (curr) {
		if (obj->vnum < curr->obj->vnum) {
			CACHE_OBJ *co = (CACHE_OBJ *) zmalloc(sizeof(CACHE_OBJ));
			co->obj = obj;
			co->cache_time = time(NULL);
			co->accessed_time = time(NULL);					
			co->next = curr;
			if (!prev) obj_cache = co;
			else prev->next = co;
			return;
		} else if (curr->obj->vnum == obj->vnum) {
			log_log(LOG_WARNING, "[CHE ] WARNING: Attempt to cache already cached object.\r\n");
			return;
		} 
		prev = curr;
		curr = curr->next;
	}

	prev->next = (CACHE_OBJ *) zmalloc(sizeof(CACHE_OBJ));
	prev->next->obj = obj;
	prev->next->cache_time = time(NULL);
	prev->next->accessed_time = time(NULL);			
	prev->next->next = 0;
}

PROP_LIST * db_gen_copy_prop_list(PROP_LIST *pl) {

	PROP_LIST *pl_new = 0;
	PROP_LIST *head = pl_new;
	PROP_LIST *pl_old = pl;
	while (pl_old) {
		if (pl_old->prop) {
			if (!pl_new) {
				head = pl_new = (PROP_LIST *) malloc(sizeof(PROP_LIST));
			} else {
				pl_new->next = (PROP_LIST *) malloc(sizeof(PROP_LIST));
				pl_new = pl_new->next;
			}
			pl_new->prop = (PROPERTY *) malloc(sizeof(PROP_LIST));
			pl_new->prop->property = make_string(pl_old->prop->property);
			pl_new->prop->value = make_string(pl_old->prop->value);
			pl_new->next = 0;
		}
		pl_old = pl_old->next;
	}

	return head;
}

void db_gen_save_obj(DATA_OBJ *obj) {
	const char *dbtype = cmfx_get_var("DBTYPE");
	if (!dbtype) dbtype = "MYSQL";

	if (!STRICMP(dbtype,"mysql")) db_mysql_save_obj(obj);
	
}

void db_gen_free_obj(DATA_OBJ *obj) {
	if (!obj) return;
	if (obj->props) {
		while(obj->props) {
			db_gen_free_property(obj->props->prop);
			PROP_LIST *p = obj->props;
			obj->props = obj->props->next;
			free(p);
		}
	}
	free(obj);
}

void db_gen_free_obj_chain(OBJECT_CHAIN *oc) {
	if (!oc) return;

	while (oc) {
		OBJECT_CHAIN *p = oc;
		oc = oc->next;
		free(p);
	}
}

OBJECT_CHAIN *db_gen_get_exits(DATA_OBJ *obj) {
	if (!obj) {
		log_log(LOG_WARNING, "[DBGN] WARNING: No object to get exits of.\r\n");
		return 0;
	}

	// If the object has no exits, return now.
	if (obj->exits == -1) return 0;

	// If the object we're looking at isn't a room, then don't bother.
	OBJECT_CHAIN *head = 0;
	OBJECT_CHAIN *oc = head = (OBJECT_CHAIN *) zmalloc(sizeof(OBJECT_CHAIN));
	
	DATA_OBJ *ob = db_gen_load_obj(obj->exits);
	while (1) {
		oc->obj = ob;
		oc->next = 0;
		if (ob->next == -1) {
			return head;
		}
		oc->next = (OBJECT_CHAIN *) zmalloc(sizeof(OBJECT_CHAIN));
		oc = oc->next;
		ob = db_gen_load_obj(ob->next);
	}

}


OBJECT_CHAIN *db_gen_get_contents(DATA_OBJ *obj) {
	if (!obj) {
		log_log(LOG_WARNING, "[DBGN] WARNING: No object to get contents of.\r\n");
		return 0;
	}

	// If the object has no contents, return now.
	if (obj->contents == -1) return 0;

	OBJECT_CHAIN *head = 0;
	OBJECT_CHAIN *oc = head = (OBJECT_CHAIN *) zmalloc(sizeof(OBJECT_CHAIN));

	DATA_OBJ *ob = db_gen_load_obj(obj->contents);
	while (1) {
		oc->obj = ob;
		oc->next = 0;
		if (ob->next == -1) {
			return head;
		}
		oc->next = (OBJECT_CHAIN *) zmalloc(sizeof(OBJECT_CHAIN));
		oc = oc->next;
		ob = db_gen_load_obj(ob->next);
	}
}

// This function is SICK. It loads all the database objects (including garbage) into memory and puts
// them in a chain for someone to play with. Very ineffecient, but required.
OBJECT_CHAIN *db_gen_get_database() {
	OBJECT_CHAIN *head = 0;
	OBJECT_CHAIN *oc = head = (OBJECT_CHAIN *) zmalloc(sizeof(OBJECT_CHAIN));

	for (int i = 0; i < db_top; i++) {
		DATA_OBJ *ob = db_gen_load_obj(i);
		oc->obj = ob;
		oc->next = (OBJECT_CHAIN *) zmalloc(sizeof(OBJECT_CHAIN));
		oc = oc->next;
	}

	return head;
}

void db_gen_recycle_obj(DATA_OBJ *obj) {

	if (!obj) {
		log_log(LOG_WARNING, "[DBGN] WARNING: Attempted to recycle null object.\r\n");
		return;
	}

	// If the object has properties, we need to delete them so the database doesn't store
	// them on save
	if (obj->props) {
		PROP_LIST *pl = obj->props;
		while (pl) {
			if (pl->prop) {
				if (pl->prop->property) free(pl->prop->property);
				if (pl->prop->value) free(pl->prop->value);
				free(pl->prop);
			}
			PROP_LIST *next = pl->next;
			free(pl);
			pl = next;
		}
		obj->props = 0;
	}

	// We're going to change the name of the object to **GARBAGE**
	strncpy(obj->name, "**GARBAGE**", 255);

	// Change its flag type to garbage, so when it cache-expires, it's saved as such.
	obj->flags |= OBJ_TYPE_GARBAGE;
}

DATA_OBJ * db_gen_create_obj(const char *name) {
	if (!name) {
		log_log(LOG_WARNING, "[DBGN] WARNING: Attempted to create non-named object.\r\n");
		return 0;
	}

	if (!strlen(name)) {
		log_log(LOG_WARNING, "[DBGN] WARNING: Attempted to create non-named object.\r\n");
		return 0;
	}

	// Check through the cached object list and see if we have a garbage item there
	DATA_OBJ *obj = 0;
	CACHE_OBJ *curr = obj_cache;
	while (curr) {
		if (curr->obj->flags & OBJ_TYPE_GARBAGE) {
			obj = curr->obj;
			// If the object has properties, we need to delete them
			if (obj->props) {
				PROP_LIST *pl = obj->props;
				while (pl) {
					if (pl->prop) {
						if (pl->prop->property) free(pl->prop->property);
						if (pl->prop->value) free(pl->prop->value);
						free(pl->prop);
					}
					PROP_LIST *next = pl->next;
					free(pl);
					pl = next;
				}
				obj->props = 0;
			}
			curr->accessed_time = time(NULL);
			break;
		}
		curr = curr->next;
	}

	// Get a garbage object and initalize to some default values. Whoever wanted
	// the object is responsible for setting flags and integrating this orphan object
	if (!obj) {
		obj = db_mysql_get_garbage_obj();
		db_gen_cache_obj(obj);
	}

	if (!obj) {
		log_log(LOG_ERROR, "[DBGN] ERROR: Could not get a valid free object.\n");
		return 0;
	}
	strncpy(obj->name, name, 255);
	obj->contents = -1;
	obj->created = (unsigned long) time(NULL);
	obj->exits = -1;
	obj->flags = OBJ_TYPE_OBJ;
	obj->home = -1;
	obj->location = -1;
	obj->modified = (unsigned long) time(NULL);
	obj->next = -1;
	obj->owner = -1;
	obj->parent = -1;
	obj->props = 0;
	obj->security = 0;
	obj->spawntime = 0;
	obj->value = -1;

	return obj;
}


