
#include "sockets.h"
#include "logging.h"
#include "utils.h"
#include "db_mysql.h"
#include "db_gen.h"
#include "objects.h"
#include "cmfx.h"
#include "commands.h"
#include "security.h"
#include "comm.h"
#include "actions.h"


int cmd_proc_commands(descrdata *d) {
	if (!d) return -1;
	if (!d->commands) return -1;
	if (!d->commands->line) return -1;

	if (!STRICMP(d->commands->line,"QUIT")) {
		d->boot = 1;
	}

	if (d->status == STATUS_WELCOMED) {
		DATA_OBJ *pc = db_gen_load_pc_name(d->commands->line);
		if (!pc) {
			sock_write(d, "\r\nThe username you entered is invalid.\r\n");
			sock_write(d, "\r\nEnter a username: ");
		} else {
			d->user = pc;
			sock_write(d, "\r\nEnter a password: ");
			d->status = STATUS_GOTUSER;
		}
	} else if (d->status == STATUS_GOTUSER) {
		PROPERTY *p =  db_gen_get_property(d->user, "@password");
		if (!p || strcmp(p->value,d->commands->line)) {
			sock_write(d, "\r\nThe password you entered is invalid.\r\n");
			sock_write(d, "Enter a username: ");
			d->status = STATUS_WELCOMED;
			d->user = 0;
		} else {
			sock_write(d,"\r\nWelcome ");
			sock_write(d,d->user->name);
			sock_write(d,"!\r\n\r\n");
			d->status = STATUS_LOGGEDIN;
			sock_addline(d,"look");
			descrdata *o = sockets;
			while (o) {
				if (o->status == STATUS_LOGGEDIN) {
					if (o != d) {
						sock_write(o,d->user->name);
						sock_write(o," has connected.\r\n");
					}
				}
				o = o->next;
			}
		}
	} else {
		// First, break it into command and argument
		char *p = d->commands->line;
		char *cmd  = 0;
		char *args = 0;

		if (*p == ':') {
			cmd  = make_string("pose");
			if (isspace(*(p + 1))) {
				args = make_string(p + 2);
			} else {
				args = make_string(p + 1);
			}
		} else if (*p == '\"') {
			cmd  = make_string("say");
			if (isspace(*(p + 1))) {
				args = make_string(p + 2);
			} else {
				args = make_string(p + 1);
			}
		} else {
			// Chomp up the whitespace
			while (isspace(*p)) p++;
			while (*p) {
				if (*p == ' ') {
					*p = '\0';
					cmd = make_string(d->commands->line);
					if (isspace(*(p + 1))) {
						args = make_string(p + 2);
					} else {
						args = make_string(p + 1);
					}
					*p = ' ';
					break;
				}
				p++;
			}
			if (!*p) {
				cmd  = make_string(d->commands->line);
				args = make_string("");
			}
		}

		switch (cmd[0]) {
		   case 'i':
		   case 'I':
			   if (!STRNICMP(cmd,"inventory",strlen(cmd))) {
				   cmd_inventory(d->user, args);
			   } else {
   			      if (!cmd_default(d->user,cmd))
				      sock_write(d,"Please enter a valid command.\r\n");
			   }
			   break;
		   case 'l':
		   case 'L':
		      if (!STRNICMP(cmd,"look",strlen(cmd))) {
		         cmd_look(d->user,args);
		      } else {
			      if (!cmd_default(d->user,cmd))
				      sock_write(d,"Please enter a valid command.\r\n");
		      }
		      break;
		      
			case 's':
			case 'S':
				if (!STRNICMP(cmd,"say",strlen(cmd))) {
					cmd_say(d->user,args);
				} else {
			      if (!cmd_default(d->user,cmd))
				      sock_write(d,"Please enter a valid command.\r\n");
				}
				break;

			case 'p':
			case 'P':
				if (!STRNICMP(cmd,"pose",strlen(cmd))) {
					cmd_pose(d->user,args);
				} else {
			      if (!cmd_default(d->user,cmd))
				      sock_write(d,"Please enter a valid command.\r\n");
				}
				break;

			case '@':
				if (!STRNICMP(cmd,"@shutdown", (3 >= strlen(cmd) ? 3 : strlen(cmd)))) {
					cmd_shutdown(d);
				} else if (!STRNICMP(cmd, "@set", (3 >= strlen(cmd) ? 3 : strlen(cmd)))) {
					cmd_set(d,args);
				} else if (!STRNICMP(cmd, "@create", strlen(cmd))) {
					cmd_create(d, args);
				} else if (!STRNICMP(cmd, "@recycle", strlen(cmd))) {
					cmd_recycle(d, args);
				} else if (!STRNICMP(cmd, "@teleport", strlen(cmd))) {
					cmd_teleport(d->user, args);
				} else {
			      if (!cmd_default(d->user,cmd))
				      sock_write(d,"Please enter a valid command.\r\n");
				}
				break;
			default:
			   if (!cmd_default(d->user,cmd))
				   sock_write(d,"Please enter a valid command.\r\n");
				break;
		}

		free(cmd);
		free(args);

	}

	inLines *l = d->commands;
	d->commands = l->next;

	free(l->line);
	free(l);

	return 0;
}

bool cmd_default(DATA_OBJ *actor, const char *cmd) {
   DATA_OBJ *action = obj_find_action(actor, cmd);
   if (!action) return false;
   
   if (action->exits != -1) {
      DATA_OBJ *dest = db_gen_load_obj(action->exits);
      DATA_OBJ *src  = db_gen_load_obj(actor->location);
      if (obj_is_room(dest)) {
         // TODO: Check the action for locks to see if we can use it

         if (act_move(actor,dest)) {
            // Tell the src we've left
            PROPERTY *pu = db_gen_get_property(action,"_success");
            PROPERTY *pt = db_gen_get_property(action,"_osuccess");
            char *youbuf = 0;
            char *thmbuf = 0;
            if (!pu) {
               youbuf = (char *) zmalloc(17);
               sprintf(youbuf,"You have left.\r\n");
			      comm_send(actor, youbuf);               
            } else {
               youbuf = (char *) zmalloc(strlen(pu->value) + 3);
               sprintf(youbuf,"%s\r\n", pu->value);
			      comm_send(actor, youbuf);                              
            }
            if (!pt) {
               thmbuf = (char *) zmalloc(strlen(actor->name) + 13);
               sprintf(thmbuf,"%s has left.\r\n",actor->name);
            } else {
               thmbuf = (char *) zmalloc(strlen(actor->name) + 1 + strlen(pu->value) + 3);
               sprintf(thmbuf,"%s %s\r\n", actor->name, pu->value);
            }
	         OBJECT_CHAIN *oc = db_gen_get_contents(src);
	         OBJECT_CHAIN *ocp = oc;
	         while (ocp) {
		         if (ocp->obj->flags & 0x00000001 || ocp->obj->flags & 0x00000002) {
		            if (ocp->obj->vnum != actor->vnum) {
		               comm_send(ocp->obj, thmbuf);
		            }
		         }
		         ocp = ocp->next;
	         }
	         free(youbuf);
	         free(thmbuf);	         
	         db_gen_free_obj_chain(oc);

            // Tell the dest we've arrived
            pu = db_gen_get_property(action,"_drop");
            pt = db_gen_get_property(action,"_odrop");
            thmbuf = 0;
            youbuf = 0;                        
            if (pu) {
               youbuf = (char *) zmalloc(strlen(pu->value) + 3);
               sprintf(youbuf,"%s\r\n", pu->value);
               comm_send(actor,youbuf);
            }
            if (!pt) {
               thmbuf = (char *) zmalloc(strlen(actor->name) + 16);
               sprintf(thmbuf,"%s has arrived.\r\n",actor->name);
            } else {
               thmbuf = (char *) zmalloc(strlen(actor->name) + 1 + strlen(pu->value) + 3);
               sprintf(thmbuf,"%s %s\r\n", actor->name, pu->value);
            }
	         oc = db_gen_get_contents(dest);
	         ocp = oc;
	         while (ocp) {
		         if (ocp->obj->flags & 0x00000001 || ocp->obj->flags & 0x00000002) {
		            if (ocp->obj->vnum != actor->vnum) {
		               comm_send(ocp->obj, thmbuf);
		            }
		         }
		         ocp = ocp->next;
	         }
	         free(thmbuf);
            if (youbuf) free(youbuf);
	         db_gen_free_obj_chain(oc);
         }
         cmd_look(actor,0);
         return true;
	  } else {
		  comm_send(actor, "You cannot take that action.\r\n");
		  return true;
	  }
   } else {
	   // TODO: Print out _fail, and _ofail messages
	   comm_send(actor, "You cannot do that.\r\n");
	   return true;
   }
   
   return false;
}

void cmd_teleport(DATA_OBJ *actor, const char *args) {
	DATA_OBJ *src  = db_gen_load_obj(actor->location);
	DATA_OBJ *dest = 0;

	if (args[0] == '#') {
		int vnum = atoi(args + 1);
		dest = db_gen_load_obj(vnum);
	}

	if (!dest) {
		comm_send(actor, "That place does not exist.\r\n");
		return;
	}
	
	if (obj_is_room(dest)) {
		// TODO: Check the action for locks to see if we can use it
		if (act_move(actor,dest)) {
			// Tell the src we've left
			char *youbuf = 0;
			char *thmbuf = 0;
			youbuf = (char *) zmalloc(17);
			sprintf(youbuf,"You have left.\r\n");
			comm_send(actor, youbuf);               
			thmbuf = (char *) zmalloc(strlen(actor->name) + 13);
			sprintf(thmbuf,"%s has left.\r\n",actor->name);

			OBJECT_CHAIN *oc = db_gen_get_contents(src);
			OBJECT_CHAIN *ocp = oc;
			while (ocp) {
				if (ocp->obj->flags & 0x00000001 || ocp->obj->flags & 0x00000002) {
				if (ocp->obj->vnum != actor->vnum) {
					comm_send(ocp->obj, thmbuf);
				}
				}
				ocp = ocp->next;
			}
			free(youbuf);
			free(thmbuf);	         
			db_gen_free_obj_chain(oc);

			// Tell the dest we've arrived
			thmbuf = 0;
			thmbuf = (char *) zmalloc(strlen(actor->name) + 16);
			sprintf(thmbuf,"%s has arrived.\r\n",actor->name);

			oc = db_gen_get_contents(dest);
			ocp = oc;
			while (ocp) {
				if (ocp->obj->flags & 0x00000001 || ocp->obj->flags & 0x00000002) {
				if (ocp->obj->vnum != actor->vnum) {
					comm_send(ocp->obj, thmbuf);
				}
				}
				ocp = ocp->next;
			}
			free(thmbuf);
			cmd_look(actor,0);
		} else {
			comm_send(actor, "You cannot go there.\r\n");
		}
	} else {
		comm_send(actor, "You can't go to anywhere but a room.\r\n");
	}
}

void cmd_inventory(DATA_OBJ *actor, const char *args) {
	bool detailed = false;

	if (args && strlen(args) && !STRNICMP(args, "detailed", strlen(args))) detailed = true;
	comm_send(actor, "Inventory:\r\n");
    if (actor->contents != -1) {
        OBJECT_CHAIN *oc_head = db_gen_get_contents(actor);
        OBJECT_CHAIN *oc = oc_head;
        OBJECT_CHAIN *curr = oc;
        while (oc) {

			// If the inventory is detailed, we need to print out each item on its own line, and unmangle
			// it for printing
			if (detailed) {
				char buf[1024];
				char * fullname = obj_unmangle(oc->obj);
				SNPRINTF(buf, 80, "   %s\r\n", fullname);
				comm_send(actor, buf);
				free(fullname);
				oc = oc->next;
				continue;
			}

			// In order to simplify things, we're going to group objects
			// with the same name together and give them a number.
			// In order to do this, we'll need to remove the nodes from the list
			// after we've determine their duplicity so we don't print 'em twice.
			curr = oc->next;
			OBJECT_CHAIN *prev = oc;
			int count = 1;
			while (curr) {
				if (!STRICMP(curr->obj->name,oc->obj->name)) {
					count++;
					prev->next = curr->next;
					free(curr);
					curr = prev->next;
				} else {
					curr = curr->next;
					prev = prev->next;
				}
			}
			if (count > 1) {
				char buf[1024];
				SNPRINTF(buf,80,"   %s (%d)\r\n",oc->obj->name,count);
				comm_send(actor,buf);
			} else {
				char buf[1024];
				SNPRINTF(buf,80,"   %s\r\n",oc->obj->name);
				comm_send(actor,buf);               
			}
			oc = oc->next;
        }
        db_gen_free_obj_chain(oc_head);
	} else {
		comm_send(actor, "   <none>\r\n");
	}
}

void cmd_recycle(descrdata *d, const char *args) {
	// TODO: Security check on the ownership of the object, as well as other permissions

	if (!args) {
		sock_write(d, "Nothing is already nothing.\r\n");
		return;
	}

	if (!strlen(args)) {
		sock_write(d, "Nothing is already nothing.\r\n");
		return;
	}

	// Eat the whitespace at the beginning
	char *myargs = make_string(args);
	char *ws = myargs;
	while (isspace(*ws)) ws++;

	// Eat the whitespace at the end
	char *ws2 = ws + strlen(ws) - 1;
	while (isspace(*ws2)) ws2--;
	*(ws2 + 1) = '\0';

	DATA_OBJ *obj = obj_match(d->user, ws);
	free(myargs);

	if (!obj) {
		sock_write(d, "Unknown or ambiguous object.\r\n");
		return;
	}

	// PC's and NPC's can't be recycled. They can only be toaded.
	if (obj_is_pc(obj) || obj_is_npc(obj)) {
		sock_write(d, "Playable objects can't be recycled.\r\n");
		return;
	}

	if (obj->vnum == 0 || obj->vnum == 1) {
		sock_write(d, "That object cannot be recycled\r\n");
		return;
	}

	// Objects need to be removed from the contents before being recycled
	if (obj_recycle(obj)) {
		sock_write(d, "Thank you for recycling.\r\n");
		return;
	}

	sock_write(d, "That object could not be recycled\r\n");

}

void cmd_create(descrdata *d, const char *args) {
	// Do some security checking on descrdata's user to see if they can create
	// objects
	// TODO: Security checking

	// Check the arguments
	if (!args) {
		sock_write(d, "You must provide a name for this object.\r\n");
		return;
	}

	if (!strlen(args)) {
		sock_write(d, "You must provide a name for this object.\r\n");
		return;
	}

	const char *p = args;
	while (*p) {
		if (*p == '=') {
			sock_write(d,"Object names may not contain an equals sign.\r\n");
			return;
		}
		p++;
	}


	// Eat the whitespace at the beginning
	char *myargs = make_string(args);
	char *ws = myargs;
	while (isspace(*ws)) ws++;

	// Eat the whitespace at the end
	char *ws2 = ws + strlen(ws) - 1;
	while (isspace(*ws2)) ws2--;
	*(ws2 + 1) = '\0';

	DATA_OBJ *obj = db_gen_create_obj(ws);
	if (!obj) {
		sock_write(d, "There was an internal error creating the object.\r\n");
		free(myargs);
		return;
	}

	// Technically the db_gen procedure inited these, but we'll do it anyway
	obj->contents = -1;
	obj->created = (unsigned long) time(NULL);
	obj->exits = -1;
	obj->flags = OBJ_TYPE_OBJ;
	obj->home = d->user->vnum;
	obj->location = d->user->vnum;
	obj->modified = (unsigned long) time(NULL);
	obj->next = -1;
	obj->owner = d->user->vnum;
	obj->parent = -1;
	obj->props = 0;
	obj->security = 0;
	obj->spawntime = 0;
	obj->value = 0;

	// Now, we need to add this object to our contents
	obj->next = d->user->contents;
	d->user->contents = obj->vnum;

	char *msg = (char *) zmalloc(7 + strlen(obj->name) + 20 + intlen(obj->vnum) + 2 + 1);
	sprintf(msg, "Object %s created with vnum #%d\r\n", obj->name, obj->vnum);
	sock_write(d, msg);

	free(myargs);

}
void cmd_set(descrdata *d, const char *args) {

	const char *ws = args;
	while (isspace(*ws)) ws++;

	char *before = make_string(ws);
   char *after = arg_sep(before,'=');

	if (!after) {
		sock_write(d,"You must provide a property name to set.\r\n");
		return;
	}

	// First, we need to find the object. Check ourselves, then check the room.
	// Don't go any further than that.

	DATA_OBJ *target = obj_match(d->user,before);
	if (!target) {
		sock_write(d,"The object you specified does not exist.\r\n");
		return;
	}

	char *prop = after;
	char *valu = arg_sep(prop,':');

	// Check the 'prop' for spaces, those are no-no's
	char *ptr = prop;
	while (*ptr) {
		if (isspace(*ptr)) {
			sock_write(d,"Spaces cannot be in the property string.\r\n");
			return;
		}
		ptr++;
	}
	
	// Check permissions on 'prop' type to make sure we're not editing values
	// we can't touch

   // Wizard-level only properties	
	if (prop[0] == '@' && d->user->security < SECURITY_GOD) {
	   sock_write(d,"You don't have permission to set that property.\r\n");
	   return;
	}
	
	// Program space only (wizards can set stuff here)
	if (prop[0] == '$' && d->user->security < SECURITY_GOD) {
	   sock_write(d,"Only programs can access this space.\r\n");
	   return;
	}
	
	// User 
	if (prop[0] == '_' && d->user->security < SECURITY_GOD && d->user->vnum != target->owner) {
	   sock_write(d,"You cannot change this property.\r\n");
	   return;
	}

	if (prop[0] == '.' && d->user->security < SECURITY_GOD && d->user->vnum != target->owner) {
	   sock_write(d,"You cannot change this property.\r\n");
	   return;
	}

	// Eat up the whitespace on the valu.
	while (isspace(*valu)) valu++;

	// If the valu is null, then we should remove the property.
	if (!valu[0]) {
		db_gen_del_property(target,prop);
		sock_write(d,"Property cleared.\r\n");
		return;
	}

	db_gen_add_property(target,prop,valu);
	sock_write(d,"Property set.\r\n");

}

void cmd_pose(DATA_OBJ *actor, const char *str) {
	DATA_OBJ *loc = db_gen_load_obj(actor->location);
	OBJECT_CHAIN *oc = db_gen_get_contents(loc);
	OBJECT_CHAIN *ocp = oc;
   char *msgbuf = (char *) zmalloc(strlen(actor->name) + strlen(str) + 4);
   sprintf(msgbuf,"%s %s\r\n",actor->name,str);
	while (ocp) {
		if (ocp->obj->flags & OBJ_TYPE_PC || ocp->obj->flags & OBJ_TYPE_NPC) {
			comm_send(ocp->obj,msgbuf);
		}
		ocp = ocp->next;
	}
	free(msgbuf);
	db_gen_free_obj_chain(oc);
}

void cmd_say(DATA_OBJ *actor, const char *str) {
	DATA_OBJ *loc = db_gen_load_obj(actor->location);
	OBJECT_CHAIN *oc = db_gen_get_contents(loc);
	OBJECT_CHAIN *ocp = oc;
   char *youbuf = (char *) zmalloc(9 + strlen(str) + 4);
   char *thmbuf = (char *) zmalloc(strlen(actor->name) + 7 + strlen(str) + 4);
   sprintf(youbuf,"You say \"%s\"\r\n",str);
   sprintf(thmbuf,"%s says \"%s\"\r\n",actor->name,str);
	while (ocp) {
		if (ocp->obj->flags & OBJ_TYPE_PC || ocp->obj->flags & OBJ_TYPE_NPC) {
			if (ocp->obj->vnum == actor->vnum) {
			   // This is an echo because it's coming back to us.
			   comm_send(ocp->obj,youbuf);
			} else {
			   comm_send(ocp->obj,thmbuf);
			}
		}
		ocp = ocp->next;
	}
	free(youbuf);
	free(thmbuf);
	db_gen_free_obj_chain(oc);
}

void cmd_shutdown(descrdata *d) {
	descrdata *o = sockets;
	while (o) {
		sock_write(o,"Server going down...\r\n\r\n");
		sock_procout(o);
		cmfx_shutdown();
		o = o->next;
	}
	log_log(LOG_NOTICE,"[CORE] NOTICE: %s has shutdown the server.\r\n",d->user->name);
}

void cmd_look(DATA_OBJ *actor, const char *args) {

   if (!args || !strlen(args)) {
      args = "here";
   }
   
   DATA_OBJ *thing = obj_match(actor,args);
   
   if (!thing) {
      comm_send(actor,"Look at what?\r\n");
      return;
   }
   
 
   // If I own the room (or have the right level, print out extra info)   
   if (thing->owner == actor->vnum || actor->security >= SECURITY_GOD) {
      char *uname = obj_unmangle(thing);
      comm_send(actor,uname);
      free(uname);
   } else {
      comm_send(actor,thing->name);   
   }
   
   comm_send(actor,"\r\n");
   
   PROPERTY * de = db_gen_get_property(thing,"_de");
   if (!de && thing->flags & 0x00000004) {
      comm_send(actor,"This place is rather non-descript.\r\n");
   } else if (!de && (thing->flags & 0x00000001 || thing->flags & 0x00000002)) {
      comm_send(actor,"That person is rather non-descript.\r\n");
   } else if (!de) {
      comm_send(actor,"That thing is rather non-descript.\r\n");
   } else {
      comm_send(actor,de->value);
      comm_send(actor, "\r\n");
   }
   
   if (!((thing->flags & 0x00000008) || (thing->flags & 0x00000010) || (thing->flags & 0x00000020))) {
      // Print exits
      if (thing->exits != -1) {
         comm_send(actor, "Exits:\r\n");
         OBJECT_CHAIN *oc_head = db_gen_get_exits(thing);
         OBJECT_CHAIN *oc = oc_head;
         OBJECT_CHAIN *curr = oc;
         while (oc) {
            char buf[1024];
            SNPRINTF(buf,80,"   %s\r\n",oc->obj->name);
            comm_send(actor,buf);               
            oc = oc->next;
         }
         db_gen_free_obj_chain(oc_head);
      }
   
      // Print contents
      if (thing->contents != -1) {
         comm_send(actor, "Contents:\r\n");
         OBJECT_CHAIN *oc_head = db_gen_get_contents(thing);
         OBJECT_CHAIN *oc = oc_head;
         OBJECT_CHAIN *curr = oc;
         while (oc) {
            // In order to simplify things, we're going to group objects
            // with the same name together and give them a number.
            // In order to do this, we'll need to remove the nodes from the list
            // after we've determine their duplicity so we don't print 'em twice.
            curr = oc->next;
            OBJECT_CHAIN *prev = oc;
            int count = 1;
            while (curr) {
               if (!STRICMP(curr->obj->name,oc->obj->name)) {
                  count++;
                  prev->next = curr->next;
                  free(curr);
                  curr = prev->next;
               } else {
                  curr = curr->next;
				  prev = prev->next;
               }
            }
            if (count > 1) {
               char buf[1024];
               SNPRINTF(buf,80,"   %s (%d)\r\n",oc->obj->name,count);
               comm_send(actor,buf);
            } else {
               char buf[1024];
               SNPRINTF(buf,80,"   %s\r\n",oc->obj->name);
               comm_send(actor,buf);               
            }
            oc = oc->next;
         }
         db_gen_free_obj_chain(oc_head);
      }
  
   }
    
   
}
